<?php
declare(strict_types=1);

require_once __DIR__ . '/../helpers/AuthHelper.php';
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../models/Eleitor.php';

class EleitoresController
{
    private $eleitorModel;

    public function __construct()
    {
        if (session_status() === PHP_SESSION_NONE)
            session_start();
        $this->eleitorModel = new Eleitor();
    }

    public function index()
    {
        AuthHelper::check();
        try {
            $termo = $_GET['search'] ?? '';
            $eleitores = !empty($termo) ? $this->eleitorModel->search($termo) : $this->eleitorModel->getAll();
            require_once __DIR__ . '/../views/eleitores/lista.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar eleitores: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/src/views/dashboard/');
            exit;
        }
    }

    public function create()
    {
        AuthHelper::check();
        require_once __DIR__ . '/../views/eleitores/cadastrar.php';
    }

    public function store()
    {
        AuthHelper::check();
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            header('Location: ' . BASE_URL . '/src/views/eleitores/cadastrar.php');
            exit;
        }

        try {
            // Validação básica
            if (empty($_POST['nome_completo']))
                throw new Exception("O nome completo é obrigatório.");
            if (empty($_POST['tipo']))
                throw new Exception("O tipo (Apoiador/Liderança) é obrigatório.");

            // Tratamento da demanda
            $demanda = $_POST['demanda'] ?? '';
            if (is_array($demanda)) {
                $demandaStr = '';
                if (!empty($demanda['categoria']))
                    $demandaStr .= "Categoria: " . $demanda['categoria'] . ". ";
                if (!empty($demanda['titulo']))
                    $demandaStr .= "Título: " . $demanda['titulo'] . ". ";
                if (!empty($demanda['descricao']))
                    $demandaStr .= "Descrição: " . $demanda['descricao'];
                $demanda = $demandaStr;
            }

            $data = [
                'nome_completo' => trim($_POST['nome_completo']),
                'cpf' => !empty($_POST['cpf']) ? preg_replace('/\D/', '', $_POST['cpf']) : null, // Remove formatação
                'data_nascimento' => $_POST['data_nascimento'] ?? null,
                'telefone' => trim($_POST['telefone'] ?? ''),
                'email' => trim($_POST['email'] ?? ''),
                'endereco' => trim($_POST['endereco'] ?? ''),
                'bairro' => trim($_POST['bairro'] ?? ''),
                'cidade' => trim($_POST['cidade'] ?? 'Belém'),
                'uf' => trim($_POST['uf'] ?? 'PA'),
                'cep' => trim($_POST['cep'] ?? ''),
                'sexo' => $_POST['sexo'] ?? null,
                'tipo' => $_POST['tipo'],
                'demanda' => trim($demanda),
                'observacao' => trim($_POST['observacao'] ?? ''),
                'usuario_id' => $_SESSION['user_id'] ?? null
            ];

            $newId = $this->eleitorModel->create($data);
            if (!$newId)
                throw new Exception('Erro ao cadastrar eleitor no banco de dados.');

            $_SESSION['success'] = 'Eleitor cadastrado com sucesso!';
            header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
            exit;

        } catch (PDOException $e) {
            // Tratamento específico para duplicidade
            if ($e->getCode() == 23000 && strpos($e->getMessage(), 'Duplicate entry') !== false) {
                if (strpos($e->getMessage(), 'cpf') !== false) {
                    $_SESSION['error'] = 'Este CPF já está cadastrado no sistema.';
                } else {
                    $_SESSION['error'] = 'Erro de duplicidade: ' . $e->getMessage();
                }
            } else {
                $_SESSION['error'] = 'Erro de banco de dados: ' . $e->getMessage();
            }
            $_SESSION['old'] = $_POST;
            header('Location: ' . BASE_URL . '/src/views/eleitores/cadastrar.php');
            exit;
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            $_SESSION['old'] = $_POST;
            header('Location: ' . BASE_URL . '/src/views/eleitores/cadastrar.php');
            exit;
        }
    }

    public function show($id)
    {
        AuthHelper::check();
        try {
            $eleitor = $this->eleitorModel->findById((int) $id);
            if (!$eleitor) {
                $_SESSION['error'] = 'Eleitor não encontrado';
                header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
                exit;
            }
            require_once __DIR__ . '/../views/eleitores/detalhes.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar eleitor: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
            exit;
        }
    }

    public function edit($id)
    {
        AuthHelper::check();
        try {
            $eleitor = $this->eleitorModel->findById((int) $id);
            if (!$eleitor) {
                $_SESSION['error'] = 'Eleitor não encontrado';
                header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
                exit;
            }
            require_once __DIR__ . '/../views/eleitores/editar.php';
        } catch (Exception $e) {
            $_SESSION['error'] = 'Erro ao carregar eleitor: ' . $e->getMessage();
            header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
            exit;
        }
    }

    public function update($id = null)
    {
        AuthHelper::check();
        if (($_SERVER['REQUEST_METHOD'] ?? 'GET') !== 'POST') {
            header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
            exit;
        }

        if ($id === null) {
            $id = $_GET['id'] ?? null;
        }

        try {
            if (!$id) {
                throw new Exception('ID do eleitor não fornecido.');
            }

            $eleitor = $this->eleitorModel->findById((int) $id);
            if (!$eleitor)
                throw new Exception('Eleitor não encontrado');

            if (empty($_POST['nome_completo']))
                throw new Exception("O nome completo é obrigatório.");
            if (empty($_POST['tipo']))
                throw new Exception("O tipo é obrigatório.");

            // Tratamento da demanda
            $demanda = $_POST['demanda'] ?? '';
            if (is_array($demanda)) {
                $demandaStr = '';
                if (!empty($demanda['categoria']))
                    $demandaStr .= "Categoria: " . $demanda['categoria'] . ". ";
                if (!empty($demanda['titulo']))
                    $demandaStr .= "Título: " . $demanda['titulo'] . ". ";
                if (!empty($demanda['descricao']))
                    $demandaStr .= "Descrição: " . $demanda['descricao'];
                $demanda = $demandaStr;
            }

            $data = [
                'nome_completo' => trim($_POST['nome_completo']),
                'cpf' => !empty($_POST['cpf']) ? preg_replace('/\D/', '', $_POST['cpf']) : null, // Remove formatação
                'data_nascimento' => $_POST['data_nascimento'] ?? null,
                'telefone' => trim($_POST['telefone'] ?? ''),
                'email' => trim($_POST['email'] ?? ''),
                'endereco' => trim($_POST['endereco'] ?? ''),
                'bairro' => trim($_POST['bairro'] ?? ''),
                'cidade' => trim($_POST['cidade'] ?? 'Belém'),
                'uf' => trim($_POST['uf'] ?? 'PA'),
                'cep' => trim($_POST['cep'] ?? ''),
                'sexo' => $_POST['sexo'] ?? null,
                'tipo' => $_POST['tipo'],
                'demanda' => trim($demanda),
                'observacao' => trim($_POST['observacao'] ?? '')
            ];

            $ok = $this->eleitorModel->update((int) $id, $data);
            if (!$ok)
                throw new Exception('Erro ao atualizar dados do eleitor.');

            $_SESSION['success'] = 'Eleitor atualizado com sucesso.';
            header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
            exit;

        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/src/views/eleitores/editar.php?id=' . $id);
            exit;
        }
    }

    public function destroy($id)
    {
        AuthHelper::check();
        try {
            $ok = $this->eleitorModel->delete((int) $id);
            if ($ok) {
                $_SESSION['success'] = 'Eleitor excluído com sucesso.';
            } else {
                $_SESSION['error'] = 'Erro ao excluir eleitor.';
            }
        } catch (Exception $e) {
            $_SESSION['error'] = $e->getMessage();
        }
        header('Location: ' . BASE_URL . '/src/views/eleitores/lista.php');
        exit;
    }
}

// Roteamento simples para chamadas diretas (se necessário, embora o ideal seja via index.php principal)
// Verifica se foi chamado diretamente e qual método executar
if (basename($_SERVER['PHP_SELF']) == 'EleitoresController.php') {
    $controller = new EleitoresController();

    // Determina a ação baseada no REQUEST_METHOD e parâmetros
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Se tiver ID na URL e for POST, pode ser update (simulado) ou store
        // Mas geralmente frameworks usam rotas. Aqui parece ser PHP puro.
        // O form de cadastro aponta para este arquivo.

        // Verifica se é update (campo hidden _method ou id na query string com ação implícita)
        // O form de cadastro não manda ID.

        // Vamos assumir: se tem id na query string e é POST -> update (ou delete se tiver _method DELETE)
        // Se não tem id -> store

        $id = $_GET['id'] ?? null;
        $action = $_GET['action'] ?? null;

        if ($action === 'delete' && $id) {
            $controller->destroy($id);
        } elseif ($id) {
            $controller->update($id);
        } else {
            $controller->store();
        }
    } elseif ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $action = $_GET['action'] ?? 'index';
        $id = $_GET['id'] ?? null;

        switch ($action) {
            case 'create':
                $controller->create();
                break;
            case 'edit':
                if ($id)
                    $controller->edit($id);
                break;
            case 'delete':
                if ($id)
                    $controller->destroy($id);
                break; // GET delete link
            default:
                $controller->index();
                break;
        }
    }
}

