<?php
require_once __DIR__ . '/../models/MaterialPublicitario.php';
require_once __DIR__ . '/../helpers/AuthHelper.php';

class MaterialPublicitarioController
{
    private $materialPublicitario;

    public function __construct()
    {
        $this->materialPublicitario = new MaterialPublicitario();
    }

    public function index()
    {
        AuthHelper::check();

        try {
            // Capturar filtros da URL
            $filtros = [
                'categoria' => $_GET['categoria'] ?? '',
                'tipo_midia' => $_GET['tipo_midia'] ?? '',
                'busca' => $_GET['busca'] ?? '',
                'campanha' => $_GET['campanha'] ?? '',
                'status' => $_GET['status'] ?? ''
            ];

            // Buscar material publicitário com filtros
            $materiais = $this->materialPublicitario->find($filtros);

            // Incluir a view
            include __DIR__ . '/../views/material_publicitario/index.php';

        } catch (Exception $e) {
            $_SESSION['erro'] = "Erro ao carregar material publicitário: " . $e->getMessage();
            include __DIR__ . '/../views/material_publicitario/index.php';
        }
    }

    public function upload()
    {
        AuthHelper::check();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                // Validar se arquivo foi enviado
                if (!isset($_FILES['arquivo']) || $_FILES['arquivo']['error'] !== UPLOAD_ERR_OK) {
                    throw new Exception('Nenhum arquivo foi enviado ou houve erro no upload.');
                }

                $arquivo = $_FILES['arquivo'];
                $titulo = $_POST['titulo'] ?? '';
                $descricao = $_POST['descricao'] ?? '';
                $categoria = $_POST['categoria'] ?? '';
                $campanha = $_POST['campanha'] ?? '';
                $publico_alvo = $_POST['publico_alvo'] ?? '';
                $tags = $_POST['tags'] ?? '';

                // Validações básicas
                if (empty($titulo)) {
                    throw new Exception('Título é obrigatório.');
                }

                if (empty($categoria)) {
                    throw new Exception('Categoria é obrigatória.');
                }

                // Definir tipo de mídia baseado no tipo do arquivo
                $tipo_midia = 'documento';
                if (strpos($arquivo['type'], 'image/') === 0) {
                    $tipo_midia = 'imagem';
                } elseif (strpos($arquivo['type'], 'video/') === 0) {
                    $tipo_midia = 'video';
                } elseif (strpos($arquivo['type'], 'audio/') === 0) {
                    $tipo_midia = 'audio';
                }

                // Gerar nome único para o arquivo
                $extensao = pathinfo($arquivo['name'], PATHINFO_EXTENSION);
                $nomeArquivo = uniqid() . '-' . preg_replace('/[^a-zA-Z0-9._-]/', '', $arquivo['name']);

                // Definir diretório de upload
                $diretorioUpload = __DIR__ . '/../../uploads/material_publicitario/';
                if (!is_dir($diretorioUpload)) {
                    mkdir($diretorioUpload, 0755, true);
                }

                $caminhoCompleto = $diretorioUpload . $nomeArquivo;
                $caminhoRelativo = 'uploads/material_publicitario/' . $nomeArquivo;

                // Mover arquivo para diretório de upload
                if (!move_uploaded_file($arquivo['tmp_name'], $caminhoCompleto)) {
                    throw new Exception('Erro ao salvar arquivo no servidor.');
                }

                // Obter dimensões para imagens
                $largura = null;
                $altura = null;
                if ($tipo_midia === 'imagem') {
                    $info = getimagesize($caminhoCompleto);
                    if ($info) {
                        $largura = $info[0];
                        $altura = $info[1];
                    }
                }

                // Preparar dados para inserção
                $dados = [
                    'id_usuario' => $_SESSION['usuario_id'],
                    'titulo' => $titulo,
                    'descricao' => $descricao,
                    'nome_arquivo' => $nomeArquivo,
                    'caminho_arquivo' => $caminhoRelativo,
                    'tipo_arquivo' => $arquivo['type'],
                    'tamanho_arquivo' => $arquivo['size'],
                    'categoria' => $categoria,
                    'tipo_midia' => $tipo_midia,
                    'largura' => $largura,
                    'altura' => $altura,
                    'campanha' => $campanha,
                    'publico_alvo' => $publico_alvo,
                    'tags' => $tags,
                    'status' => 'rascunho'
                ];

                // Inserir no banco de dados
                $id = $this->materialPublicitario->create($dados);

                $_SESSION['sucesso'] = 'Material publicitário enviado com sucesso!';
                header('Location: ' . BASE_URL . '/material-publicitario');
                exit;

            } catch (Exception $e) {
                $_SESSION['erro'] = $e->getMessage();
                header('Location: ' . BASE_URL . '/material-publicitario');
                exit;
            }
        }
    }

    public function visualizar($id)
    {
        AuthHelper::check();

        try {
            $material = $this->materialPublicitario->findById($id);

            if (!$material) {
                throw new Exception('Material publicitário não encontrado.');
            }

            // Incrementar contador de visualizações
            $this->materialPublicitario->incrementarVisualizacao($id);

            // Incluir view de visualização
            include __DIR__ . '/../views/material_publicitario/visualizar.php';

        } catch (Exception $e) {
            $_SESSION['erro'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/material-publicitario');
            exit;
        }
    }

    public function download($id)
    {
        AuthHelper::check();

        try {
            $material = $this->materialPublicitario->findById($id);

            if (!$material) {
                throw new Exception('Material publicitário não encontrado.');
            }

            $caminhoArquivo = __DIR__ . '/../../' . $material['caminho_arquivo'];

            if (!file_exists($caminhoArquivo)) {
                throw new Exception('Arquivo não encontrado no servidor.');
            }

            // Incrementar contador de downloads
            $this->materialPublicitario->incrementarDownload($id);

            // Configurar headers para download
            header('Content-Type: ' . $material['tipo_arquivo']);
            header('Content-Disposition: attachment; filename="' . $material['nome_arquivo'] . '"');
            header('Content-Length: ' . filesize($caminhoArquivo));

            // Enviar arquivo
            readfile($caminhoArquivo);
            exit;

        } catch (Exception $e) {
            $_SESSION['erro'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/material-publicitario');
            exit;
        }
    }

    public function editar($id)
    {
        AuthHelper::check();

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            try {
                $dados = [
                    'titulo' => $_POST['titulo'] ?? '',
                    'descricao' => $_POST['descricao'] ?? '',
                    'categoria' => $_POST['categoria'] ?? '',
                    'tipo_midia' => $_POST['tipo_midia'] ?? '',
                    'campanha' => $_POST['campanha'] ?? '',
                    'data_evento' => $_POST['data_evento'] ?? null,
                    'publico_alvo' => $_POST['publico_alvo'] ?? '',
                    'tags' => $_POST['tags'] ?? '',
                    'cor_predominante' => $_POST['cor_predominante'] ?? '',
                    'formato_final' => $_POST['formato_final'] ?? '',
                    'status' => $_POST['status'] ?? 'rascunho'
                ];

                $this->materialPublicitario->update($id, $dados);

                $_SESSION['sucesso'] = 'Material publicitário atualizado com sucesso!';
                header('Location: ' . BASE_URL . '/material-publicitario');
                exit;

            } catch (Exception $e) {
                $_SESSION['erro'] = $e->getMessage();
            }
        }

        try {
            $material = $this->materialPublicitario->findById($id);

            if (!$material) {
                throw new Exception('Material publicitário não encontrado.');
            }

            include __DIR__ . '/../views/material_publicitario/editar.php';

        } catch (Exception $e) {
            $_SESSION['erro'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/material-publicitario');
            exit;
        }
    }

    public function excluir($id)
    {
        AuthHelper::check();

        try {
            $material = $this->materialPublicitario->findById($id);

            if (!$material) {
                throw new Exception('Material publicitário não encontrado.');
            }

            // Excluir arquivo físico
            $caminhoArquivo = __DIR__ . '/../../' . $material['caminho_arquivo'];
            if (file_exists($caminhoArquivo)) {
                unlink($caminhoArquivo);
            }

            // Excluir do banco de dados
            $this->materialPublicitario->delete($id);

            $_SESSION['sucesso'] = 'Material publicitário excluído com sucesso!';

        } catch (Exception $e) {
            $_SESSION['erro'] = $e->getMessage();
        }

        header('Location: ' . BASE_URL . '/material-publicitario');
        exit;
    }

    public function estatisticas()
    {
        AuthHelper::check();

        try {
            $estatisticas = $this->materialPublicitario->getEstatisticas();

            include __DIR__ . '/../views/material_publicitario/estatisticas.php';

        } catch (Exception $e) {
            $_SESSION['erro'] = $e->getMessage();
            header('Location: ' . BASE_URL . '/material-publicitario');
            exit;
        }
    }
}
?>