<?php
class Agenda
{
    private $pdo;

    public function __construct()
    {
        try {
            $this->pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $this->pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            $this->ensureSchema();
            $this->updateSchema();
        } catch (PDOException $e) {
            die("Erro na conexão com o banco de dados: " . $e->getMessage());
        }
    }

    private function ensureSchema()
    {
        $sql = "CREATE TABLE IF NOT EXISTS agendas (
            id INT NOT NULL AUTO_INCREMENT,
            id_eleitor INT DEFAULT NULL,
            titulo VARCHAR(255) NOT NULL,
            descricao TEXT,
            categoria ENUM('Saúde','Jurídico','Assistência Social','Moradia','Visita','Ação Social','Ação Saúde','Fiscalização','Reunião') NOT NULL,
            data_inicio DATETIME NOT NULL,
            data_fim DATETIME DEFAULT NULL,
            id_responsavel INT DEFAULT NULL,
            google_calendar_event_id VARCHAR(255) DEFAULT NULL,
            whatsapp_notificacao TINYINT(1) DEFAULT 0,
            data_criacao TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_general_ci";
        $this->pdo->exec($sql);
    }

    private function updateSchema()
    {
        // Adicionar coluna whatsapp_notificacao se não existir
        try {
            $this->pdo->query("SELECT whatsapp_notificacao FROM agendas LIMIT 1");
        } catch (PDOException $e) {
            $this->pdo->exec("ALTER TABLE agendas ADD COLUMN whatsapp_notificacao TINYINT(1) DEFAULT 0");
        }

        // Adicionar coluna google_calendar_event_id se não existir
        try {
            $this->pdo->query("SELECT google_calendar_event_id FROM agendas LIMIT 1");
        } catch (PDOException $e) {
            $this->pdo->exec("ALTER TABLE agendas ADD COLUMN google_calendar_event_id VARCHAR(255) DEFAULT NULL");
        }
    }

    public function getAgendasPorIntervalo($start, $end)
    {
        $sql = "SELECT id, titulo as title, data_inicio as start, data_fim as end FROM agendas WHERE data_inicio BETWEEN :start AND :end";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':start' => $start, ':end' => $end]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    public function create($dados)
    {
        $sql = "INSERT INTO agendas (id_eleitor, titulo, descricao, categoria, data_inicio, data_fim, id_responsavel, whatsapp_notificacao)
                VALUES (:id_eleitor, :titulo, :descricao, :categoria, :data_inicio, :data_fim, :id_responsavel, :whatsapp_notificacao)";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([
            ':id_eleitor' => !empty($dados['id_eleitor']) ? $dados['id_eleitor'] : null,
            ':titulo' => $dados['titulo'],
            ':descricao' => $dados['descricao'],
            ':categoria' => $dados['categoria'],
            ':data_inicio' => $dados['data_inicio'],
            ':data_fim' => $dados['data_fim'],
            ':id_responsavel' => $dados['id_responsavel'],
            ':whatsapp_notificacao' => $dados['whatsapp_notificacao'] ?? 0
        ]);
        return $this->pdo->lastInsertId();
    }

    /**
     * NOVO MÉTODO: Atualiza um agendamento com o ID do evento do Google.
     */
    public function updateGoogleEventId($id_agendamento, $google_event_id)
    {
        $sql = "UPDATE agendas SET google_calendar_event_id = :google_event_id WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':google_event_id' => $google_event_id,
            ':id' => $id_agendamento
        ]);
    }

    /**
     * Busca um agendamento específico pelo ID
     */
    public function getById($id)
    {
        $sql = "SELECT a.*, e.nome_completo as eleitor_nome 
                FROM agendas a 
                LEFT JOIN eleitores e ON a.id_eleitor = e.id 
                WHERE a.id = :id";
        $stmt = $this->pdo->prepare($sql);
        $stmt->execute([':id' => $id]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    }

    /**
     * Atualiza um agendamento existente
     */
    public function update($id, $dados)
    {
        $sql = "UPDATE agendas SET 
                    id_eleitor = :id_eleitor,
                    titulo = :titulo,
                    descricao = :descricao,
                    categoria = :categoria,
                    data_inicio = :data_inicio,
                    data_fim = :data_fim,
                    whatsapp_notificacao = :whatsapp_notificacao
                WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([
            ':id' => $id,
            ':id_eleitor' => !empty($dados['id_eleitor']) ? $dados['id_eleitor'] : null,
            ':titulo' => $dados['titulo'],
            ':descricao' => $dados['descricao'],
            ':categoria' => $dados['categoria'],
            ':data_inicio' => $dados['data_inicio'],
            ':data_fim' => $dados['data_fim'],
            ':whatsapp_notificacao' => $dados['whatsapp_notificacao'] ?? 0
        ]);
    }

    /**
     * Exclui um agendamento
     */
    public function delete($id)
    {
        $sql = "DELETE FROM agendas WHERE id = :id";
        $stmt = $this->pdo->prepare($sql);
        return $stmt->execute([':id' => $id]);
    }
}
