<?php
// /gabinete/src/views/documentos/upload.php
if (session_status() === PHP_SESSION_NONE) { session_start(); }

require_once __DIR__ . '/../../config/config.php';
require_once __DIR__ . '/../../models/Documento.php';

// Protege rota
if (empty($_SESSION['user_id'])) {
    header('Location: ' . BASE_URL . '/login.php');
    exit;
}

$documento = new Documento();
$mensagem = '';
$erro = '';
$docExistente = null;
$isEdicao = false;

// Verificar se é edição
$id = intval($_GET['id'] ?? 0);
if ($id > 0) {
    $isEdicao = true;
    try {
        $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        $stmt = $pdo->prepare("SELECT * FROM documentos WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $docExistente = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$docExistente) {
            $erro = 'Documento não encontrado.';
            $isEdicao = false;
        }
    } catch (Exception $e) {
        $erro = 'Erro ao carregar documento: ' . $e->getMessage();
        $isEdicao = false;
    }
}

// Processar upload ou edição
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        $titulo = trim($_POST['titulo'] ?? '');
        $descricao = trim($_POST['descricao'] ?? '');
        $categoria = $_POST['categoria'] ?? 'Diversos';
        
        // Validações básicas
        if (empty($titulo)) {
            throw new Exception('Título é obrigatório');
        }
        
        if ($isEdicao) {
            // Modo edição - apenas atualizar metadados
            $pdo = new PDO('mysql:host=' . DB_HOST . ';dbname=' . DB_NAME, DB_USER, DB_PASS);
            $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
            
            $stmt = $pdo->prepare("
                UPDATE documentos 
                SET titulo = :titulo, descricao = :descricao, categoria = :categoria 
                WHERE id = :id
            ");
            
            if ($stmt->execute([
                'titulo' => $titulo,
                'descricao' => $descricao,
                'categoria' => $categoria,
                'id' => $id
            ])) {
                $mensagem = 'Documento atualizado com sucesso!';
                // Recarregar dados do documento
                $stmt = $pdo->prepare("SELECT * FROM documentos WHERE id = :id");
                $stmt->execute(['id' => $id]);
                $docExistente = $stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                throw new Exception('Erro ao atualizar documento');
            }
        } else {
            // Modo criação - upload de novo arquivo
            $arquivo = $_FILES['arquivo'];
            
            if ($arquivo['error'] !== UPLOAD_ERR_OK) {
                throw new Exception('Erro no upload do arquivo');
            }
            
            // Determinar tipo de mídia
            $tipoMidia = $documento->determinarTipoMidia($arquivo['name']);
            
            // Validar arquivo
            $validacao = $documento->validarArquivo($arquivo, $tipoMidia);
            if (!$validacao['valido']) {
                throw new Exception($validacao['erro']);
            }
            
            // Criar diretório de upload se não existir
            $diretorioUpload = __DIR__ . '/../../../uploads/documentos/' . date('Y/m');
            if (!is_dir($diretorioUpload)) {
                mkdir($diretorioUpload, 0755, true);
            }
            
            // Gerar nome único para o arquivo
            $extensao = pathinfo($arquivo['name'], PATHINFO_EXTENSION);
            $nomeArquivo = uniqid() . '_' . time() . '.' . $extensao;
            $caminhoCompleto = $diretorioUpload . '/' . $nomeArquivo;
            $caminhoRelativo = 'uploads/documentos/' . date('Y/m') . '/' . $nomeArquivo;
            
            // Mover arquivo
            if (!move_uploaded_file($arquivo['tmp_name'], $caminhoCompleto)) {
                throw new Exception('Erro ao salvar arquivo');
            }
            
            // Obter informações adicionais para imagens
            $largura = null;
            $altura = null;
            if ($tipoMidia === 'imagem') {
                $infoImagem = $documento->obterInfoImagem($caminhoCompleto);
                $largura = $infoImagem['largura'];
                $altura = $infoImagem['altura'];
            }
            
            // Salvar no banco de dados
            $dadosDocumento = [
                'id_usuario' => $_SESSION['user_id'],
                'titulo' => $titulo,
                'descricao' => $descricao,
                'nome_arquivo' => $arquivo['name'],
                'caminho_arquivo' => $caminhoRelativo,
                'tipo_arquivo' => $arquivo['type'],
                'tamanho_arquivo' => $arquivo['size'],
                'categoria' => $categoria,
                'tipo_midia' => $tipoMidia,
                'largura' => $largura,
                'altura' => $altura
            ];
            
            if ($documento->create($dadosDocumento)) {
                $mensagem = 'Arquivo enviado com sucesso!';
                // Limpar campos após sucesso
                $_POST = [];
            } else {
                throw new Exception('Erro ao salvar no banco de dados');
            }
        }
        
    } catch (Exception $e) {
        $erro = $e->getMessage();
        // Remover arquivo se foi criado mas houve erro no banco
        if (isset($caminhoCompleto) && file_exists($caminhoCompleto)) {
            unlink($caminhoCompleto);
        }
    }
}

$nome = $_SESSION['nome'] ?? 'Usuário';
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?= $isEdicao ? 'Editar Documento' : 'Upload de Documentos' ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #0B1F3A;
            --secondary-color: #102B54;
            --accent-color: #3B82F6;
        }
        
        .navbar-custom {
            background: linear-gradient(135deg, var(--primary-color), var(--secondary-color));
        }
        
        .upload-area {
            border: 2px dashed #dee2e6;
            border-radius: 12px;
            padding: 40px;
            text-align: center;
            background: #f8f9fa;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .upload-area:hover {
            border-color: var(--accent-color);
            background: #e3f2fd;
        }
        
        .upload-area.dragover {
            border-color: var(--accent-color);
            background: #e3f2fd;
            transform: scale(1.02);
        }
        
        .file-preview {
            max-width: 200px;
            max-height: 200px;
            border-radius: 8px;
            margin: 10px auto;
        }
        
        .file-info {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
        }
        
        .progress-container {
            display: none;
            margin-top: 15px;
        }
        
        .supported-formats {
            font-size: 0.875rem;
            color: #6c757d;
            margin-top: 10px;
        }
    </style>
</head>
<body>
    <!-- Navbar -->
    <nav class="navbar navbar-expand-lg navbar-dark navbar-custom">
        <div class="container-fluid">
            <span class="navbar-brand">
                <i class="<?= $isEdicao ? 'bi bi-pencil-fill' : 'bi bi-cloud-upload-fill' ?> me-2"></i>
                <?= $isEdicao ? 'Editar Documento' : 'Upload de Documentos' ?>
            </span>
            <div class="d-flex align-items-center text-white">
                <span class="me-3">Olá, <?= htmlspecialchars($nome) ?></span>
                <a href="lista.php" class="btn btn-outline-light btn-sm me-2">
                    <i class="bi bi-list-ul me-1"></i>Lista
                </a>
                <a href="<?= BASE_URL ?>/src/views/dashboard/" class="btn btn-outline-light btn-sm">
                    <i class="bi bi-house-fill me-1"></i>Painel
                </a>
            </div>
        </div>
    </nav>

    <div class="container py-4">
        <!-- Mensagens -->
        <?php if ($mensagem): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <i class="bi bi-check-circle-fill me-2"></i>
                <?= htmlspecialchars($mensagem) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>
        
        <?php if ($erro): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <i class="bi bi-exclamation-triangle-fill me-2"></i>
                <?= htmlspecialchars($erro) ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="row justify-content-center">
            <div class="col-lg-8">
                <div class="card shadow">
                    <div class="card-header">
                        <h5 class="mb-0">
                            <i class="bi bi-cloud-upload me-2"></i>
                            Enviar Novo Documento
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST" enctype="multipart/form-data" id="uploadForm">
                            <!-- Área de Upload -->
                            <div class="upload-area" id="uploadArea">
                                <i class="bi bi-cloud-upload display-1 text-muted mb-3"></i>
                                <h5>Clique aqui ou arraste arquivos</h5>
                                <p class="text-muted mb-0">Selecione os arquivos que deseja enviar</p>
                                <input type="file" class="d-none" id="arquivo" name="arquivo" accept=".pdf,.doc,.docx,.txt,.odt,.jpg,.jpeg,.png,.gif,.webp,.svg,.mp4,.avi,.mov,.wmv,.flv,.webm" required>
                                
                                <div class="supported-formats">
                                    <strong>Formatos suportados:</strong><br>
                                    <span class="badge bg-primary me-1">Documentos</span> PDF, DOC, DOCX, TXT, ODT<br>
                                    <span class="badge bg-success me-1">Imagens</span> JPG, PNG, GIF, WebP, SVG<br>
                                    <span class="badge bg-warning me-1">Vídeos</span> MP4, AVI, MOV, WMV, WebM
                                </div>
                            </div>
                            
                            <!-- Preview do arquivo -->
                            <div id="filePreview" class="d-none">
                                <div class="file-info">
                                    <div class="row align-items-center">
                                        <div class="col-auto">
                                            <div id="previewIcon" class="text-center">
                                                <i class="bi bi-file-earmark display-4"></i>
                                            </div>
                                        </div>
                                        <div class="col">
                                            <h6 id="fileName" class="mb-1"></h6>
                                            <small id="fileSize" class="text-muted"></small>
                                            <div id="fileDimensions" class="text-muted small"></div>
                                        </div>
                                        <div class="col-auto">
                                            <button type="button" class="btn btn-outline-danger btn-sm" id="removeFile">
                                                <i class="bi bi-trash"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <!-- Barra de progresso -->
                            <div class="progress-container">
                                <div class="progress">
                                    <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                                </div>
                            </div>
                            
                            <!-- Campos do formulário -->
                            <div class="row mt-4">
                                <div class="col-md-8">
                                    <label for="titulo" class="form-label">Título *</label>
                                    <input type="text" class="form-control" id="titulo" name="titulo" 
                                           value="<?= htmlspecialchars($_POST['titulo'] ?? '') ?>" required>
                                </div>
                                <div class="col-md-4">
                                    <label for="categoria" class="form-label">Categoria</label>
                                    <select class="form-select" id="categoria" name="categoria">
                                        <option value="Diversos" <?= ($_POST['categoria'] ?? '') === 'Diversos' ? 'selected' : '' ?>>Diversos</option>
                                        <option value="Ofício" <?= ($_POST['categoria'] ?? '') === 'Ofício' ? 'selected' : '' ?>>Ofício</option>
                                        <option value="Requerimento" <?= ($_POST['categoria'] ?? '') === 'Requerimento' ? 'selected' : '' ?>>Requerimento</option>
                                        <option value="Projetos" <?= ($_POST['categoria'] ?? '') === 'Projetos' ? 'selected' : '' ?>>Projetos</option>
                                        <option value="Prestação de Contas" <?= ($_POST['categoria'] ?? '') === 'Prestação de Contas' ? 'selected' : '' ?>>Prestação de Contas</option>
                                        <option value="Foto Publicitária" <?= ($_POST['categoria'] ?? '') === 'Foto Publicitária' ? 'selected' : '' ?>>Foto Publicitária</option>
                                        <option value="Vídeo Publicitário" <?= ($_POST['categoria'] ?? '') === 'Vídeo Publicitário' ? 'selected' : '' ?>>Vídeo Publicitário</option>
                                        <option value="Material de Campanha" <?= ($_POST['categoria'] ?? '') === 'Material de Campanha' ? 'selected' : '' ?>>Material de Campanha</option>
                                    </select>
                                </div>
                            </div>
                            
                            <div class="mt-3">
                                <label for="descricao" class="form-label">Descrição</label>
                                <textarea class="form-control" id="descricao" name="descricao" rows="3" 
                                          placeholder="Descrição opcional do documento..."><?= htmlspecialchars($_POST['descricao'] ?? '') ?></textarea>
                            </div>
                            
                            <!-- Botões -->
                            <div class="d-flex justify-content-between mt-4">
                                <a href="lista.php" class="btn btn-outline-secondary">
                                    <i class="bi bi-arrow-left me-1"></i>Voltar
                                </a>
                                <button type="submit" class="btn btn-primary" id="submitBtn">
                                    <i class="bi bi-cloud-upload me-1"></i>Enviar Arquivo
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        document.addEventListener('DOMContentLoaded', function() {
            const uploadArea = document.getElementById('uploadArea');
            const fileInput = document.getElementById('arquivo');
            const filePreview = document.getElementById('filePreview');
            const fileName = document.getElementById('fileName');
            const fileSize = document.getElementById('fileSize');
            const fileDimensions = document.getElementById('fileDimensions');
            const previewIcon = document.getElementById('previewIcon');
            const removeFileBtn = document.getElementById('removeFile');
            const tituloInput = document.getElementById('titulo');
            
            // Clique na área de upload
            uploadArea.addEventListener('click', () => fileInput.click());
            
            // Drag and drop
            uploadArea.addEventListener('dragover', (e) => {
                e.preventDefault();
                uploadArea.classList.add('dragover');
            });
            
            uploadArea.addEventListener('dragleave', () => {
                uploadArea.classList.remove('dragover');
            });
            
            uploadArea.addEventListener('drop', (e) => {
                e.preventDefault();
                uploadArea.classList.remove('dragover');
                
                const files = e.dataTransfer.files;
                if (files.length > 0) {
                    fileInput.files = files;
                    handleFileSelect(files[0]);
                }
            });
            
            // Seleção de arquivo
            fileInput.addEventListener('change', (e) => {
                if (e.target.files.length > 0) {
                    handleFileSelect(e.target.files[0]);
                }
            });
            
            // Remover arquivo
            removeFileBtn.addEventListener('click', () => {
                fileInput.value = '';
                filePreview.classList.add('d-none');
                uploadArea.style.display = 'block';
            });
            
            function handleFileSelect(file) {
                // Mostrar preview
                fileName.textContent = file.name;
                fileSize.textContent = formatBytes(file.size);
                
                // Auto-preencher título se estiver vazio
                if (!tituloInput.value) {
                    const nameWithoutExt = file.name.replace(/\.[^/.]+$/, "");
                    tituloInput.value = nameWithoutExt;
                }
                
                // Determinar ícone baseado no tipo
                const extension = file.name.split('.').pop().toLowerCase();
                let icon = 'bi-file-earmark';
                let iconClass = 'text-primary';
                
                if (['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'].includes(extension)) {
                    icon = 'bi-image';
                    iconClass = 'text-success';
                    
                    // Mostrar preview da imagem
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        const img = document.createElement('img');
                        img.src = e.target.result;
                        img.className = 'file-preview';
                        img.onload = () => {
                            fileDimensions.textContent = `${img.naturalWidth} × ${img.naturalHeight} pixels`;
                        };
                        previewIcon.innerHTML = '';
                        previewIcon.appendChild(img);
                    };
                    reader.readAsDataURL(file);
                } else if (['mp4', 'avi', 'mov', 'wmv', 'flv', 'webm'].includes(extension)) {
                    icon = 'bi-camera-video';
                    iconClass = 'text-warning';
                } else if (['mp3', 'wav', 'ogg'].includes(extension)) {
                    icon = 'bi-music-note';
                    iconClass = 'text-info';
                } else if (['pdf'].includes(extension)) {
                    icon = 'bi-file-earmark-pdf';
                    iconClass = 'text-danger';
                }
                
                if (!['jpg', 'jpeg', 'png', 'gif', 'webp', 'svg'].includes(extension)) {
                    previewIcon.innerHTML = `<i class="bi ${icon} display-4 ${iconClass}"></i>`;
                }
                
                uploadArea.style.display = 'none';
                filePreview.classList.remove('d-none');
            }
            
            function formatBytes(bytes, decimals = 2) {
                if (bytes === 0) return '0 Bytes';
                const k = 1024;
                const dm = decimals < 0 ? 0 : decimals;
                const sizes = ['Bytes', 'KB', 'MB', 'GB'];
                const i = Math.floor(Math.log(bytes) / Math.log(k));
                return parseFloat((bytes / Math.pow(k, i)).toFixed(dm)) + ' ' + sizes[i];
            }
        });
    </script>
</body>
</html>